<?php
/**
 * Plugin Name: Elementor Quiz Spam Protection
 * Description: Fügt ein Quiz-Feld zu Elementor Forms hinzu als eleganten Spam-Schutz
 * Version: 1.0.2
 * Author: fastWP
 * Text Domain: fastwp.de
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Elementor tested up to: 3.18
 */

if (!defined('ABSPATH')) {
    exit;
}

define('EQSP_VERSION', '1.0.0');
define('EQSP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('EQSP_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Hauptklasse des Plugins
 */
final class Elementor_Quiz_Spam_Protection {

    private static $instance = null;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('plugins_loaded', [$this, 'init']);
    }

    public function init() {
        // Prüfen ob Elementor Pro aktiv ist
        if (!defined('ELEMENTOR_PRO_VERSION')) {
            add_action('admin_notices', [$this, 'admin_notice_missing_elementor_pro']);
            return;
        }

        // Admin Settings laden
        require_once EQSP_PLUGIN_DIR . 'includes/class-admin-settings.php';
        new EQSP_Admin_Settings();

        // Elementor Field registrieren
        add_action('elementor_pro/forms/fields/register', [$this, 'register_quiz_field']);
        
        // Validierung registrieren
        add_action('elementor_pro/forms/validation', [$this, 'validate_quiz_field'], 10, 2);
        
        // Assets laden
        add_action('admin_enqueue_scripts', [$this, 'admin_assets']);
        add_action('wp_enqueue_scripts', [$this, 'frontend_assets']);
    }

    /**
     * Admin Notice wenn Elementor Pro fehlt
     */
    public function admin_notice_missing_elementor_pro() {
        $message = sprintf(
            '<strong>%s</strong> benötigt <strong>Elementor Pro</strong> um zu funktionieren.',
            'Elementor Quiz Spam Protection'
        );
        printf('<div class="notice notice-error"><p>%s</p></div>', $message);
    }

    /**
     * Quiz Field registrieren
     */
    public function register_quiz_field($form_fields_registrar) {
        require_once EQSP_PLUGIN_DIR . 'includes/class-elementor-quiz-field.php';
        $form_fields_registrar->register(new EQSP_Elementor_Quiz_Field());
    }

    /**
     * Quiz Antwort validieren
     */
    public function validate_quiz_field($record, $ajax_handler) {
        $fields = $record->get_field([
            'type' => 'quiz_spam'
        ]);

        if (empty($fields)) {
            return;
        }

        foreach ($fields as $field) {
            $field_id = $field['id'];
            $submitted_answer = strtolower(trim($field['value']));
            
            // Hash aus POST-Daten auslesen
            $hash_field_name = $field_id . '_hash';
            $correct_answer_hash = '';
            
            // Direkt aus $_POST holen
            if (isset($_POST['form_fields'][$hash_field_name])) {
                $correct_answer_hash = sanitize_text_field($_POST['form_fields'][$hash_field_name]);
            }
            
            // Validiere die Antwort
            if (empty($correct_answer_hash) || !$this->verify_answer($submitted_answer, $correct_answer_hash)) {
                $error_message = !empty($field['quiz_error_message']) 
                    ? $field['quiz_error_message'] 
                    : __('Die Antwort auf die Sicherheitsfrage ist falsch.', 'eq-spam-protection');
                
                $ajax_handler->add_error($field_id, $error_message);
            }
        }
    }

    /**
     * Antwort verifizieren
     */
    private function verify_answer($submitted, $hash) {
        // Hash enthält: base64(question_index|correct_answer_hash|timestamp)
        $decoded = base64_decode($hash);
        if (!$decoded) {
            return false;
        }
        
        $parts = explode('|', $decoded);
        if (count($parts) !== 3) {
            return false;
        }
        
        $correct_hash = $parts[1];
        $timestamp = intval($parts[2]);
        
        // Zeitlimit prüfen (30 Minuten)
        if (time() - $timestamp > 1800) {
            return false;
        }
        
        // Antwort prüfen
        return hash_equals($correct_hash, wp_hash($submitted));
    }

    /**
     * Admin Assets
     */
    public function admin_assets($hook) {
        if ($hook === 'settings_page_eqsp-settings') {
            wp_enqueue_style(
                'eqsp-admin',
                EQSP_PLUGIN_URL . 'assets/css/admin.css',
                [],
                EQSP_VERSION
            );
            wp_enqueue_script(
                'eqsp-admin',
                EQSP_PLUGIN_URL . 'assets/js/admin.js',
                ['jquery'],
                EQSP_VERSION,
                true
            );
        }
    }

    /**
     * Frontend Assets
     */
    public function frontend_assets() {
        wp_enqueue_style(
            'eqsp-frontend',
            EQSP_PLUGIN_URL . 'assets/css/frontend.css',
            [],
            EQSP_VERSION
        );
    }
}

// Plugin initialisieren
Elementor_Quiz_Spam_Protection::instance();
