<?php
/**
 * Admin Settings für globale Quiz-Fragen
 */

if (!defined('ABSPATH')) {
    exit;
}

class EQSP_Admin_Settings {

    private $option_name = 'eqsp_global_questions';

    public function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_eqsp_save_questions', [$this, 'ajax_save_questions']);
    }

    /**
     * Settings-Seite hinzufügen
     */
    public function add_settings_page() {
        add_options_page(
            __('Quiz Spam Protection', 'eq-spam-protection'),
            __('Quiz Spam Protection', 'eq-spam-protection'),
            'manage_options',
            'eqsp-settings',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Settings registrieren
     */
    public function register_settings() {
        register_setting($this->option_name, $this->option_name, [
            'sanitize_callback' => [$this, 'sanitize_questions']
        ]);
    }

    /**
     * Fragen sanitizen
     */
    public function sanitize_questions($input) {
        if (!is_array($input)) {
            return [];
        }

        $sanitized = [];
        foreach ($input as $item) {
            if (!empty($item['question']) && !empty($item['answer'])) {
                $sanitized[] = [
                    'question' => sanitize_text_field($item['question']),
                    'answer' => strtolower(sanitize_text_field($item['answer']))
                ];
            }
        }

        return $sanitized;
    }

    /**
     * Globale Fragen abrufen
     */
    public static function get_global_questions() {
        $questions = get_option('eqsp_global_questions', []);
        
        // Default-Fragen wenn leer
        if (empty($questions)) {
            $questions = [
                ['question' => 'Wieviel ist 3 + 4?', 'answer' => '7'],
                ['question' => 'Welche Farbe hat der Himmel?', 'answer' => 'blau'],
                ['question' => 'Hauptstadt von Deutschland?', 'answer' => 'berlin'],
            ];
        }
        
        return $questions;
    }

    /**
     * Settings-Seite rendern
     */
    public function render_settings_page() {
        $questions = self::get_global_questions();
        ?>
        <div class="wrap eqsp-settings">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="eqsp-intro">
                <p><?php _e('Definiere hier deine globalen Quiz-Fragen für den Spam-Schutz. Diese können in jedem Elementor-Formular verwendet werden.', 'eq-spam-protection'); ?></p>
            </div>

            <form method="post" action="options.php" id="eqsp-questions-form">
                <?php settings_fields($this->option_name); ?>
                
                <div class="eqsp-questions-wrapper">
                    <div class="eqsp-questions-header">
                        <h2><?php _e('Quiz-Fragen', 'eq-spam-protection'); ?></h2>
                        <span class="eqsp-question-count">
                            <span id="eqsp-count"><?php echo count($questions); ?></span>/10 <?php _e('Fragen', 'eq-spam-protection'); ?>
                        </span>
                    </div>

                    <div id="eqsp-questions-list">
                        <?php 
                        $index = 0;
                        foreach ($questions as $q) : 
                        ?>
                        <div class="eqsp-question-row" data-index="<?php echo $index; ?>">
                            <div class="eqsp-question-number"><?php echo $index + 1; ?></div>
                            <div class="eqsp-question-fields">
                                <div class="eqsp-field">
                                    <label><?php _e('Frage', 'eq-spam-protection'); ?></label>
                                    <input type="text" 
                                           name="<?php echo $this->option_name; ?>[<?php echo $index; ?>][question]" 
                                           value="<?php echo esc_attr($q['question']); ?>" 
                                           placeholder="z.B. Wieviel ist 2 + 3?" />
                                </div>
                                <div class="eqsp-field eqsp-field-answer">
                                    <label><?php _e('Antwort', 'eq-spam-protection'); ?></label>
                                    <input type="text" 
                                           name="<?php echo $this->option_name; ?>[<?php echo $index; ?>][answer]" 
                                           value="<?php echo esc_attr($q['answer']); ?>" 
                                           placeholder="z.B. 5" />
                                </div>
                            </div>
                            <button type="button" class="eqsp-remove-question" title="<?php _e('Frage entfernen', 'eq-spam-protection'); ?>">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </div>
                        <?php 
                        $index++;
                        endforeach; 
                        ?>
                    </div>

                    <button type="button" id="eqsp-add-question" class="button button-secondary" <?php echo count($questions) >= 10 ? 'disabled' : ''; ?>>
                        <span class="dashicons dashicons-plus-alt2"></span>
                        <?php _e('Frage hinzufügen', 'eq-spam-protection'); ?>
                    </button>
                </div>

                <div class="eqsp-tips">
                    <h3><?php _e('Tipps für gute Quiz-Fragen', 'eq-spam-protection'); ?></h3>
                    <ul>
                        <li><?php _e('Verwende einfache Fragen, die jeder beantworten kann', 'eq-spam-protection'); ?></li>
                        <li><?php _e('Vermeide zu komplizierte Rechenaufgaben', 'eq-spam-protection'); ?></li>
                        <li><?php _e('Die Antwort sollte eindeutig sein (z.B. "blau" statt "himmelblau")', 'eq-spam-protection'); ?></li>
                        <li><?php _e('Groß-/Kleinschreibung wird automatisch ignoriert', 'eq-spam-protection'); ?></li>
                        <li><?php _e('Mindestens 3-5 Fragen für gute Variation', 'eq-spam-protection'); ?></li>
                    </ul>
                </div>

                <?php submit_button(__('Fragen speichern', 'eq-spam-protection')); ?>
            </form>
        </div>

        <!-- Template für neue Frage -->
        <script type="text/template" id="eqsp-question-template">
            <div class="eqsp-question-row" data-index="{{INDEX}}">
                <div class="eqsp-question-number">{{NUMBER}}</div>
                <div class="eqsp-question-fields">
                    <div class="eqsp-field">
                        <label><?php _e('Frage', 'eq-spam-protection'); ?></label>
                        <input type="text" 
                               name="<?php echo $this->option_name; ?>[{{INDEX}}][question]" 
                               value="" 
                               placeholder="z.B. Wieviel ist 2 + 3?" />
                    </div>
                    <div class="eqsp-field eqsp-field-answer">
                        <label><?php _e('Antwort', 'eq-spam-protection'); ?></label>
                        <input type="text" 
                               name="<?php echo $this->option_name; ?>[{{INDEX}}][answer]" 
                               value="" 
                               placeholder="z.B. 5" />
                    </div>
                </div>
                <button type="button" class="eqsp-remove-question" title="<?php _e('Frage entfernen', 'eq-spam-protection'); ?>">
                    <span class="dashicons dashicons-trash"></span>
                </button>
            </div>
        </script>
        <?php
    }
}
