<?php
/*
Plugin Name:       Content Statistiken
Plugin URI:        https://fastwp.de/alle-woerter-auf-wordpress-seite-zaehlen/  // ➊
Description:       Zeigt Wort- und Zeichenzahlen im Dashboard und über jedem Artikel
Version:           1.0
Author:            fastWP
Author URI:        https://fastwp.de                                // ➋
Update URI:        https://fastwp.de/alle-woerter-auf-wordpress-seite-zaehlen/   // ➌
License:           GPL-2.0-or-later
License URI:       https://www.gnu.org/licenses/gpl-2.0.html
Text Domain:       content-statistiken
Requires at least: 5.8
Requires PHP:      7.4
*/


if (!defined('ABSPATH')) exit;

// Frontend-Funktionen laden
require_once __DIR__ . '/frontend-counter.php';

// Dashboard-Statistiken
add_action('wp_dashboard_setup', 'content_statistiken_dashboard_widget');

function content_statistiken_dashboard_widget() {
    wp_add_dashboard_widget(
        'content_statistiken_widget',
        'Content-Statistiken',
        'content_statistiken_render'
    );
}

function content_statistiken_get_data() {
    $stats = get_transient('content_statistiken_data');

    if (!$stats) {
        $stats = [
            'post_types' => [],
            'total' => [
                'words' => 0,
                'chars' => 0,
                'items' => 0
            ]
        ];

        $post_types = get_post_types(['public' => true], 'objects');
        
        foreach ($post_types as $post_type) {
            $posts = get_posts([
                'post_type' => $post_type->name,
                'posts_per_page' => -1,
                'post_status' => 'publish'
            ]);

            $type_stats = [
                'name' => $post_type->labels->name,
                'words' => 0,
                'chars' => 0,
                'count' => 0
            ];

            foreach ($posts as $post) {
                $text = strip_tags($post->post_content);
                $text = preg_replace('/\s+/', ' ', trim($text));
                
                preg_match_all('/\p{L}+/u', $text, $matches);
                $words = count($matches[0]);
                $chars = mb_strlen(preg_replace('/\s+/', '', $text));

                $type_stats['words'] += $words;
                $type_stats['chars'] += $chars;
                $type_stats['count']++;

                $stats['total']['words'] += $words;
                $stats['total']['chars'] += $chars;
                $stats['total']['items']++;
            }

            if ($type_stats['count'] > 0) {
                $stats['post_types'][$post_type->name] = $type_stats;
            }
        }

        set_transient('content_statistiken_data', $stats, 12 * HOUR_IN_SECONDS);
    }

    return $stats;
}

function content_statistiken_render() {
    $stats = content_statistiken_get_data();
    ?>
    <div class="content-stats">
        <style>
            .content-stats table { width:100%; border-collapse:collapse; margin-bottom:15px; }
            .content-stats th, .content-stats td { padding:10px; border-bottom:1px solid #eee; }
            .content-stats tr:nth-child(even) { background:#f9f9f9; }
            .content-stats .total-row { background:#f1f1f1; font-weight:600; }
        </style>

        <table>
            <thead>
                <tr>
                    <th>Inhaltstyp</th>
                    <th>Anzahl</th>
                    <th>Wörter</th>
                    <th>Zeichen</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($stats['post_types'] as $type) : ?>
                <tr>
                    <td><?= esc_html($type['name']) ?></td>
                    <td><?= number_format($type['count'], 0, ',', '.') ?></td>
                    <td><?= number_format($type['words'], 0, ',', '.') ?></td>
                    <td><?= number_format($type['chars'], 0, ',', '.') ?></td>
                </tr>
                <?php endforeach; ?>
                <tr class="total-row">
                    <td>Gesamt</td>
                    <td><?= number_format($stats['total']['items'], 0, ',', '.') ?></td>
                    <td><?= number_format($stats['total']['words'], 0, ',', '.') ?></td>
                    <td><?= number_format($stats['total']['chars'], 0, ',', '.') ?></td>
                </tr>
            </tbody>
        </table>

        <div style="color:#666; font-size:13px; padding:10px; border-top:1px solid #eee; margin-top:15px;">
            🕒 Letzte Aktualisierung: <?= date('d.m.Y H:i') ?><br>
            <small>Automatische Aktualisierung alle 12 Stunden</small>
        </div>
    </div>
    <?php
}

// Cache leeren bei Änderungen
add_action('save_post', 'content_statistiken_clear_cache');
function content_statistiken_clear_cache($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    delete_transient('content_statistiken_data');
}